<?php

declare(strict_types=1);

const LOCAL_IMAGES_DIR = __DIR__ . '/images';
const LOCAL_IMAGES_PREFIX = 'images/';

function jsonResponse(array $payload, int $status = 200): void
{
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($payload);
    exit;
}

function readJsonBody(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || trim($raw) === '') {
        return [];
    }

    $data = json_decode($raw, true);
    if (!is_array($data)) {
        jsonResponse(['message' => 'Invalid JSON body'], 400);
    }

    return $data;
}

function setupCors(): void
{
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, PUT, PATCH, DELETE, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');

    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(204);
        exit;
    }
}

function getBearerToken(): ?string
{
    $header = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    if (preg_match('/Bearer\s+(.*)$/i', $header, $matches) !== 1) {
        return null;
    }

    return trim($matches[1]);
}

function parseCarWriteRequest(string $method): array
{
    $rawContentType = trim((string)($_SERVER['CONTENT_TYPE'] ?? ''));
    $normalizedContentType = strtolower($rawContentType);

    if (str_starts_with($normalizedContentType, 'multipart/form-data')) {
        if (strtoupper($method) === 'POST') {
            return [
                'body' => normalizeRequestBody($_POST),
                'files' => normalizePhpFilesArray($_FILES),
            ];
        }

        return parseMultipartBodyFromInput($rawContentType);
    }

    if (str_starts_with($normalizedContentType, 'application/json')) {
        return [
            'body' => normalizeRequestBody(readJsonBody()),
            'files' => [],
        ];
    }

    if (strtoupper($method) === 'POST') {
        return [
            'body' => normalizeRequestBody($_POST),
            'files' => normalizePhpFilesArray($_FILES),
        ];
    }

    $raw = file_get_contents('php://input');
    if ($raw === false || trim($raw) === '') {
        return ['body' => [], 'files' => []];
    }

    parse_str($raw, $parsed);

    return [
        'body' => normalizeRequestBody(is_array($parsed) ? $parsed : []),
        'files' => [],
    ];
}

function parseMultipartBodyFromInput(string $contentType): array
{
    if (preg_match('/boundary=(?:"([^"]+)"|([^;]+))/i', $contentType, $matches) !== 1) {
        jsonResponse([
            'message' => 'Validation failed',
            'errors' => ['request' => ['Invalid multipart boundary.']],
        ], 422);
    }

    $boundary = (string)($matches[1] !== '' ? $matches[1] : $matches[2]);
    $raw = file_get_contents('php://input');

    if ($raw === false) {
        jsonResponse([
            'message' => 'Validation failed',
            'errors' => ['request' => ['Unable to read multipart input.']],
        ], 422);
    }

    $delimiter = '--' . $boundary;
    $parts = explode($delimiter, $raw);

    $fields = [];
    $files = [];

    foreach ($parts as $part) {
        $part = ltrim($part, "\r\n");
        $part = rtrim($part, "\r\n");

        if ($part === '' || $part === '--') {
            continue;
        }

        if (str_ends_with($part, '--')) {
            $part = substr($part, 0, -2);
        }

        $sections = explode("\r\n\r\n", $part, 2);
        if (count($sections) !== 2) {
            continue;
        }

        [$rawHeaders, $content] = $sections;
        $headers = parseMultipartHeaders($rawHeaders);
        $contentDisposition = $headers['content-disposition'] ?? '';

        if (preg_match('/name="([^"]+)"/', $contentDisposition, $nameMatches) !== 1) {
            continue;
        }

        $fieldName = $nameMatches[1];
        $content = rtrim($content, "\r\n");

        if (preg_match('/filename="([^"]*)"/', $contentDisposition, $fileMatches) !== 1 || $fileMatches[1] === '') {
            assignMultipartFieldValue($fields, $fieldName, $content);
            continue;
        }

        $tmpPath = tempnam(sys_get_temp_dir(), 'upload_');
        if ($tmpPath === false || file_put_contents($tmpPath, $content) === false) {
            jsonResponse([
                'message' => 'Validation failed',
                'errors' => ['images' => ['Failed to process uploaded file.']],
            ], 422);
        }

        assignMultipartFileValue($files, $fieldName, [
            'name' => $fileMatches[1],
            'type' => (string)($headers['content-type'] ?? 'application/octet-stream'),
            'tmp_name' => $tmpPath,
            'error' => UPLOAD_ERR_OK,
            'size' => strlen($content),
        ]);
    }

    return [
        'body' => normalizeRequestBody($fields),
        'files' => $files,
    ];
}

function parseMultipartHeaders(string $rawHeaders): array
{
    $headers = [];

    foreach (explode("\r\n", $rawHeaders) as $line) {
        $line = trim($line);
        if ($line === '' || !str_contains($line, ':')) {
            continue;
        }

        [$name, $value] = explode(':', $line, 2);
        $headers[strtolower(trim($name))] = trim($value);
    }

    return $headers;
}

function assignMultipartFieldValue(array &$fields, string $name, string $value): void
{
    $normalizedName = normalizeFieldName($name);

    if (str_ends_with($name, '[]')) {
        if (!isset($fields[$normalizedName]) || !is_array($fields[$normalizedName])) {
            $fields[$normalizedName] = [];
        }

        $fields[$normalizedName][] = $value;
        return;
    }

    $fields[$normalizedName] = $value;
}

function assignMultipartFileValue(array &$files, string $name, array $file): void
{
    $normalizedName = normalizeFieldName($name);

    if (!isset($files[$normalizedName])) {
        $files[$normalizedName] = [];
    }

    $files[$normalizedName][] = $file;
}

function normalizeFieldName(string $name): string
{
    return str_ends_with($name, '[]') ? substr($name, 0, -2) : $name;
}

function normalizeRequestBody(array $body): array
{
    $normalized = [];

    foreach ($body as $key => $value) {
        if (is_string($value)) {
            $normalized[$key] = trim($value);
            continue;
        }

        $normalized[$key] = $value;
    }

    if (array_key_exists('images', $normalized) && is_string($normalized['images'])) {
        $decoded = json_decode($normalized['images'], true);
        if (is_array($decoded)) {
            $normalized['images'] = $decoded;
        }
    }

    return $normalized;
}

function normalizePhpFilesArray(array $files): array
{
    $normalized = [];

    foreach ($files as $field => $spec) {
        if (!is_array($spec) || !isset($spec['name'], $spec['type'], $spec['tmp_name'], $spec['error'], $spec['size'])) {
            continue;
        }

        $entries = [];
        flattenPhpFileSpec($spec['name'], $spec['type'], $spec['tmp_name'], $spec['error'], $spec['size'], $entries);

        foreach ($entries as $entry) {
            $normalizedField = normalizeFieldName((string)$field);
            if (!isset($normalized[$normalizedField])) {
                $normalized[$normalizedField] = [];
            }

            $normalized[$normalizedField][] = $entry;
        }
    }

    return $normalized;
}

function flattenPhpFileSpec(mixed $name, mixed $type, mixed $tmpName, mixed $error, mixed $size, array &$entries): void
{
    if (is_array($name)) {
        foreach ($name as $index => $nestedName) {
            flattenPhpFileSpec(
                $nestedName,
                $type[$index] ?? null,
                $tmpName[$index] ?? null,
                $error[$index] ?? null,
                $size[$index] ?? null,
                $entries
            );
        }

        return;
    }

    if ((int)$error === UPLOAD_ERR_NO_FILE || (string)$tmpName === '') {
        return;
    }

    $entries[] = [
        'name' => (string)$name,
        'type' => (string)$type,
        'tmp_name' => (string)$tmpName,
        'error' => (int)$error,
        'size' => (int)$size,
    ];
}

function getPreferredCoverUpload(array $files): ?array
{
    foreach (['image', 'cover_image'] as $field) {
        if (!isset($files[$field]) || !is_array($files[$field])) {
            continue;
        }

        foreach ($files[$field] as $file) {
            if (isValidUploadedFile($file)) {
                return $file;
            }
        }
    }

    return null;
}

function getGalleryUploads(array $files): array
{
    $uploads = [];

    foreach (['images', 'gallery_images'] as $field) {
        if (!isset($files[$field]) || !is_array($files[$field])) {
            continue;
        }

        foreach ($files[$field] as $file) {
            if (isValidUploadedFile($file)) {
                $uploads[] = $file;
            }
        }
    }

    return $uploads;
}

function isValidUploadedFile(mixed $file): bool
{
    if (!is_array($file)) {
        return false;
    }

    return isset($file['tmp_name'], $file['error']) && (int)$file['error'] === UPLOAD_ERR_OK && (string)$file['tmp_name'] !== '';
}

function validateCreateCarPayload(array $body, bool $hasCoverUpload): array
{
    $requiredTextFields = [
        'make',
        'model',
        'year',
        'vin',
        'mileage',
        'price',
        'location',
        'status',
        'title_status',
        'damage_summary',
        'description',
    ];

    $errors = [];

    foreach ($requiredTextFields as $field) {
        if (!array_key_exists($field, $body) || trim((string)$body[$field]) === '') {
            $errors[$field][] = 'This field is required.';
        }
    }

    if (!$hasCoverUpload) {
        if (!array_key_exists('image_url', $body) || trim((string)$body['image_url']) === '') {
            $errors['image_url'][] = 'This field is required when no cover image file is provided.';
        } elseif (!isImageReference((string)$body['image_url'])) {
            $errors['image_url'][] = 'Must be an image URL, data URL, or local images path.';
        }
    } elseif (isset($body['image_url']) && trim((string)$body['image_url']) !== '' && !isImageReference((string)$body['image_url'])) {
        $errors['image_url'][] = 'Must be an image URL, data URL, or local images path.';
    }

    $images = validateImageObjectsField($body, $errors);

    if ($errors !== []) {
        jsonResponse(['message' => 'Validation failed', 'errors' => $errors], 422);
    }

    return [
        'make' => trim((string)$body['make']),
        'model' => trim((string)$body['model']),
        'year' => (int)$body['year'],
        'vin' => trim((string)$body['vin']),
        'mileage' => (int)$body['mileage'],
        'price' => (float)$body['price'],
        'location' => trim((string)$body['location']),
        'status' => trim((string)$body['status']),
        'title_status' => trim((string)$body['title_status']),
        'damage_summary' => trim((string)$body['damage_summary']),
        'description' => trim((string)$body['description']),
        'image_url' => trim((string)($body['image_url'] ?? '')),
        'images' => $images,
        '_images_provided' => array_key_exists('images', $body),
    ];
}

function validateUpdateCarPayload(array $body, bool $hasCoverUpload, bool $hasGalleryUploads): array
{
    $scalarFields = [
        'make',
        'model',
        'vin',
        'location',
        'status',
        'title_status',
        'damage_summary',
        'description',
        'image_url',
    ];

    $numericFields = ['year', 'mileage', 'price'];
    $errors = [];

    foreach ($scalarFields as $field) {
        if (!array_key_exists($field, $body)) {
            continue;
        }

        if (trim((string)$body[$field]) === '') {
            $errors[$field][] = 'This field is required.';
        }
    }

    foreach ($numericFields as $field) {
        if (!array_key_exists($field, $body)) {
            continue;
        }

        if (trim((string)$body[$field]) === '') {
            $errors[$field][] = 'This field is required.';
        }
    }

    if (array_key_exists('image_url', $body) && trim((string)$body['image_url']) !== '' && !isImageReference((string)$body['image_url'])) {
        $errors['image_url'][] = 'Must be an image URL, data URL, or local images path.';
    }

    $images = validateImageObjectsField($body, $errors);

    if ($errors !== []) {
        jsonResponse(['message' => 'Validation failed', 'errors' => $errors], 422);
    }

    $payload = [
        'images' => $images,
        '_images_provided' => array_key_exists('images', $body),
    ];

    foreach ($scalarFields as $field) {
        if (!array_key_exists($field, $body)) {
            continue;
        }

        $payload[$field] = trim((string)$body[$field]);
    }

    foreach ($numericFields as $field) {
        if (!array_key_exists($field, $body)) {
            continue;
        }

        $payload[$field] = $field === 'price' ? (float)$body[$field] : (int)$body[$field];
    }

    $hasTextualInput = false;
    foreach (array_merge($scalarFields, $numericFields) as $field) {
        if (array_key_exists($field, $payload)) {
            $hasTextualInput = true;
            break;
        }
    }

    if (!$hasTextualInput && !$hasCoverUpload && !$hasGalleryUploads && !$payload['_images_provided']) {
        jsonResponse([
            'message' => 'Validation failed',
            'errors' => ['request' => ['No update fields were provided.']],
        ], 422);
    }

    return $payload;
}

function validateImageObjectsField(array $body, array &$errors): array
{
    if (!array_key_exists('images', $body)) {
        return [];
    }

    $value = $body['images'];

    if (!is_array($value)) {
        $errors['images'][] = 'Must be an array of objects with a url field.';
        return [];
    }

    $normalized = [];

    foreach ($value as $index => $image) {
        if (!is_array($image)) {
            $errors["images.$index"][] = 'Must be an object.';
            continue;
        }

        if (!array_key_exists('url', $image) || trim((string)$image['url']) === '') {
            $errors["images.$index.url"][] = 'This field is required.';
            continue;
        }

        $url = trim((string)$image['url']);
        if (!isImageReference($url)) {
            $errors["images.$index.url"][] = 'Must be an image URL, data URL, or local images path.';
            continue;
        }

        $normalized[] = ['url' => $url];
    }

    return $normalized;
}

function materializeImageReferenceList(array $images, array &$createdFiles): array
{
    $materialized = [];

    foreach ($images as $image) {
        $materialized[] = [
            'url' => materializeImageReference((string)$image['url'], $createdFiles),
        ];
    }

    return $materialized;
}

function materializeUploadedImages(array $uploads, array &$createdFiles): array
{
    $materialized = [];

    foreach ($uploads as $file) {
        $materialized[] = [
            'url' => materializeUploadedFile($file, $createdFiles),
        ];
    }

    return $materialized;
}

function materializeUploadedFile(array $file, array &$createdFiles): string
{
    if (!isValidUploadedFile($file)) {
        throw new RuntimeException('Invalid uploaded image.');
    }

    $tmpPath = (string)$file['tmp_name'];
    $binary = @file_get_contents($tmpPath);
    if ($binary === false || $binary === '') {
        throw new RuntimeException('Uploaded image could not be read.');
    }

    $relativePath = writeImageBinary($binary, detectImageExtension($binary));
    $createdFiles[] = $relativePath;

    return $relativePath;
}

function isImageReference(string $value): bool
{
    if (isLocalImagePath($value)) {
        return true;
    }

    if (preg_match('#^data:image/[a-zA-Z0-9.+-]+;base64,#', $value) === 1) {
        return true;
    }

    return filter_var($value, FILTER_VALIDATE_URL) !== false;
}

function materializeImageReference(string $value, array &$createdFiles): string
{
    $value = trim($value);

    if (isLocalImagePath($value)) {
        return normalizeLocalImagePath($value);
    }

    if (preg_match('#^data:image/([a-zA-Z0-9.+-]+);base64,(.+)$#', $value, $matches) === 1) {
        $binary = base64_decode($matches[2], true);
        if ($binary === false) {
            throw new RuntimeException('One or more images contain invalid base64 data.');
        }

        $extension = extensionFromMimeSubtype(strtolower($matches[1]));
        $relativePath = writeImageBinary($binary, $extension);
        $createdFiles[] = $relativePath;

        return $relativePath;
    }

    if (filter_var($value, FILTER_VALIDATE_URL) !== false) {
        $binary = fetchRemoteImage($value);
        $extension = detectImageExtension($binary);
        $relativePath = writeImageBinary($binary, $extension);
        $createdFiles[] = $relativePath;

        return $relativePath;
    }

    throw new RuntimeException('One or more images are not a valid URL, data URL, or local images path.');
}

function fetchRemoteImage(string $url): string
{
    $context = stream_context_create([
        'http' => [
            'timeout' => 20,
            'follow_location' => 1,
        ],
        'ssl' => [
            'verify_peer' => true,
            'verify_peer_name' => true,
        ],
    ]);

    $binary = @file_get_contents($url, false, $context);
    if ($binary === false || $binary === '') {
        throw new RuntimeException('One or more image URLs could not be downloaded.');
    }

    return $binary;
}

function detectImageExtension(string $binary): string
{
    if (!function_exists('finfo_open')) {
        return 'jpeg';
    }

    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    if ($finfo === false) {
        return 'jpeg';
    }

    $mimeType = finfo_buffer($finfo, $binary) ?: '';
    finfo_close($finfo);

    return match ($mimeType) {
        'image/jpeg' => 'jpeg',
        'image/png' => 'jpeg',
        'image/webp' => 'jpeg',
        'image/gif' => 'jpeg',
        default => 'jpeg',
    };
}

function extensionFromMimeSubtype(string $subtype): string
{
    return match ($subtype) {
        'jpeg', 'jpg' => 'jpeg',
        'png' => 'jpeg',
        'webp' => 'jpeg',
        'gif' => 'jpeg',
        default => 'jpeg',
    };
}

function writeImageBinary(string $binary, string $extension): string
{
    ensureImagesDirectory();

    $filename = 'car_' . bin2hex(random_bytes(8)) . '.' . $extension;
    $relativePath = LOCAL_IMAGES_PREFIX . $filename;
    $absolutePath = __DIR__ . '/' . $relativePath;

    if (@file_put_contents($absolutePath, $binary) === false) {
        throw new RuntimeException('Failed to store one or more images on server.');
    }

    return $relativePath;
}

function ensureImagesDirectory(): void
{
    if (is_dir(LOCAL_IMAGES_DIR)) {
        return;
    }

    if (!@mkdir(LOCAL_IMAGES_DIR, 0775, true) && !is_dir(LOCAL_IMAGES_DIR)) {
        throw new RuntimeException('Failed to prepare image storage directory.');
    }
}

function isLocalImagePath(string $path): bool
{
    $normalized = normalizeLocalImagePath($path);

    return str_starts_with($normalized, LOCAL_IMAGES_PREFIX);
}

function normalizeLocalImagePath(string $path): string
{
    $trimmed = trim($path);

    if (str_starts_with($trimmed, '/')) {
        $trimmed = ltrim($trimmed, '/');
    }

    return $trimmed;
}

function fetchCar(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM cars WHERE id = :id LIMIT 1');
    $stmt->execute([':id' => $id]);
    $car = $stmt->fetch();

    if (!$car) {
        return null;
    }

    $imagesStmt = $pdo->prepare('SELECT url FROM car_images WHERE car_id = :car_id ORDER BY sort_order ASC, id ASC');
    $imagesStmt->execute([':car_id' => $id]);
    $images = $imagesStmt->fetchAll();

    $car['images'] = array_map(
        static fn (array $row): array => ['url' => (string)$row['url']],
        $images
    );

    return $car;
}

function replaceCarImages(PDO $pdo, int $carId, array $images): void
{
    $deleteStmt = $pdo->prepare('DELETE FROM car_images WHERE car_id = :car_id');
    $deleteStmt->execute([':car_id' => $carId]);

    if ($images === []) {
        return;
    }

    $insertStmt = $pdo->prepare('INSERT INTO car_images (car_id, url, sort_order) VALUES (:car_id, :url, :sort_order)');

    foreach ($images as $index => $image) {
        $insertStmt->execute([
            ':car_id' => $carId,
            ':url' => $image['url'],
            ':sort_order' => $index,
        ]);
    }
}

function collectCarImagePaths(?array $car): array
{
    if (!$car) {
        return [];
    }

    $paths = [];

    if (isset($car['image_url']) && isLocalImagePath((string)$car['image_url'])) {
        $paths[] = normalizeLocalImagePath((string)$car['image_url']);
    }

    if (isset($car['images']) && is_array($car['images'])) {
        foreach ($car['images'] as $image) {
            if (isset($image['url']) && isLocalImagePath((string)$image['url'])) {
                $paths[] = normalizeLocalImagePath((string)$image['url']);
            }
        }
    }

    return array_values(array_unique($paths));
}

function deleteLocalImageFiles(array $paths): void
{
    foreach (array_unique($paths) as $relativePath) {
        if (!is_string($relativePath) || !isLocalImagePath($relativePath)) {
            continue;
        }

        $absolutePath = __DIR__ . '/' . normalizeLocalImagePath($relativePath);
        if (is_file($absolutePath)) {
            @unlink($absolutePath);
        }
    }
}

function deleteUnreferencedLocalImageFiles(PDO $pdo, array $paths): void
{
    $checkCarStmt = $pdo->prepare('SELECT COUNT(*) FROM cars WHERE image_url = :path');
    $checkGalleryStmt = $pdo->prepare('SELECT COUNT(*) FROM car_images WHERE url = :path');

    foreach (array_unique($paths) as $path) {
        if (!is_string($path) || !isLocalImagePath($path)) {
            continue;
        }

        $normalizedPath = normalizeLocalImagePath($path);

        $checkCarStmt->execute([':path' => $normalizedPath]);
        $carCount = (int)$checkCarStmt->fetchColumn();

        $checkGalleryStmt->execute([':path' => $normalizedPath]);
        $galleryCount = (int)$checkGalleryStmt->fetchColumn();

        if ($carCount === 0 && $galleryCount === 0) {
            deleteLocalImageFiles([$normalizedPath]);
        }
    }
}
