<?php

declare(strict_types=1);

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/helpers.php';

const DEFAULT_IMAGE_PATH = 'images/default-placeholder.jpeg';

function db(): PDO
{
    static $pdo = null;

    if ($pdo instanceof PDO) {
        return $pdo;
    }

    $isNew = !file_exists(DB_PATH);

    $pdo = new PDO('sqlite:' . DB_PATH);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    $pdo->exec('PRAGMA foreign_keys = ON');

    if ($isNew) {
        initializeDatabase($pdo);
    }

    migrateDatabase($pdo);

    return $pdo;
}

function initializeDatabase(PDO $pdo): void
{
    $pdo->exec('CREATE TABLE admins (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        email TEXT NOT NULL UNIQUE,
        password_hash TEXT NOT NULL,
        created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    )');

    $pdo->exec('CREATE TABLE admin_tokens (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        admin_id INTEGER NOT NULL,
        token TEXT NOT NULL UNIQUE,
        expires_at TEXT NOT NULL,
        created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY(admin_id) REFERENCES admins(id) ON DELETE CASCADE
    )');

    $pdo->exec('CREATE TABLE cars (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        make TEXT NOT NULL,
        model TEXT NOT NULL,
        year INTEGER NOT NULL,
        vin TEXT NOT NULL UNIQUE,
        mileage INTEGER NOT NULL,
        price REAL NOT NULL,
        location TEXT NOT NULL,
        status TEXT NOT NULL,
        title_status TEXT NOT NULL,
        damage_summary TEXT NOT NULL,
        description TEXT NOT NULL,
        image_url TEXT NOT NULL,
        created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    )');

    $pdo->exec('CREATE TABLE car_images (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        car_id INTEGER NOT NULL,
        url TEXT NOT NULL,
        sort_order INTEGER NOT NULL DEFAULT 0,
        created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY(car_id) REFERENCES cars(id) ON DELETE CASCADE
    )');

    $adminEmail = 'admin@salvageyard.com';
    $adminPasswordHash = password_hash('admin123', PASSWORD_DEFAULT);

    $stmt = $pdo->prepare('INSERT INTO admins (email, password_hash) VALUES (:email, :password_hash)');
    $stmt->execute([
        ':email' => $adminEmail,
        ':password_hash' => $adminPasswordHash,
    ]);

    $cars = seedCars();

    $carStmt = $pdo->prepare('INSERT INTO cars (make, model, year, vin, mileage, price, location, status, title_status, damage_summary, description, image_url)
    VALUES (:make, :model, :year, :vin, :mileage, :price, :location, :status, :title_status, :damage_summary, :description, :image_url)');

    $imageStmt = $pdo->prepare('INSERT INTO car_images (car_id, url, sort_order) VALUES (:car_id, :url, :sort_order)');

    foreach ($cars as $car) {
        $primaryImage = storeSeedImageOrFallback($car['image_url']);

        $carStmt->execute([
            ':make' => $car['make'],
            ':model' => $car['model'],
            ':year' => $car['year'],
            ':vin' => $car['vin'],
            ':mileage' => $car['mileage'],
            ':price' => $car['price'],
            ':location' => $car['location'],
            ':status' => $car['status'],
            ':title_status' => $car['title_status'],
            ':damage_summary' => $car['damage_summary'],
            ':description' => $car['description'],
            ':image_url' => $primaryImage,
        ]);

        $carId = (int)$pdo->lastInsertId();

        foreach ($car['images'] as $index => $url) {
            $imageStmt->execute([
                ':car_id' => $carId,
                ':url' => storeSeedImageOrFallback($url),
                ':sort_order' => $index,
            ]);
        }
    }
}

function migrateDatabase(PDO $pdo): void
{
    if (!tableExists($pdo, 'cars')) {
        return;
    }

    if (!columnExists($pdo, 'cars', 'image_url')) {
        $pdo->exec("ALTER TABLE cars ADD COLUMN image_url TEXT NOT NULL DEFAULT ''");
    }

    if (!tableExists($pdo, 'car_images')) {
        $pdo->exec('CREATE TABLE car_images (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            car_id INTEGER NOT NULL,
            url TEXT NOT NULL,
            sort_order INTEGER NOT NULL DEFAULT 0,
            created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY(car_id) REFERENCES cars(id) ON DELETE CASCADE
        )');
    }

    ensureDefaultPlaceholderImage();

    $stmt = $pdo->prepare("UPDATE cars SET image_url = :default_image WHERE image_url IS NULL OR TRIM(image_url) = ''");
    $stmt->execute([':default_image' => DEFAULT_IMAGE_PATH]);

    populateSeedImageFilesForExistingCars($pdo);
}

function seedCars(): array
{
    return [
        [
            'make' => 'Toyota',
            'model' => 'Camry',
            'year' => 2018,
            'vin' => '4T1B11HK6JU123456',
            'mileage' => 84200,
            'price' => 4900,
            'location' => 'Nairobi, Kenya',
            'status' => 'Available',
            'title_status' => 'Salvage',
            'damage_summary' => 'Front bumper and hood damage',
            'description' => 'Runs and drives. Engine and transmission are in good condition.',
            'image_url' => 'https://images.unsplash.com/photo-1550355291-bbee04a92027?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1503376780353-7e6692767b70?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1493238792000-8113da705763?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1494976388531-d1058494cdd8?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Ford',
            'model' => 'F-150',
            'year' => 2020,
            'vin' => '1FTEW1E42LFA65432',
            'mileage' => 67100,
            'price' => 8600,
            'location' => 'Mombasa, Kenya',
            'status' => 'Available',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Right side panel damage',
            'description' => 'Solid frame, clean interior, airbag deployed previously.',
            'image_url' => 'https://images.unsplash.com/photo-1511919884226-fd3cad34687c?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1485291571150-772bcfc10da5?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1489824904134-891ab64532f1?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1492144534655-ae79c964c9d7?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Honda',
            'model' => 'Civic',
            'year' => 2017,
            'vin' => '2HGFC2F59HH987654',
            'mileage' => 93500,
            'price' => 3900,
            'location' => 'Kisumu, Kenya',
            'status' => 'Pending',
            'title_status' => 'Salvage',
            'damage_summary' => 'Rear collision damage',
            'description' => 'Starts, shifts smoothly, minor suspension work needed.',
            'image_url' => 'https://images.unsplash.com/photo-1617814076367-b759c7d7e738?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1517673132405-a56a62b18caf?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1549399542-7e3f8b79c341?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1590362891991-f776e747a588?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Nissan',
            'model' => 'X-Trail',
            'year' => 2019,
            'vin' => 'JN8AT2MT5KW007101',
            'mileage' => 78300,
            'price' => 7450,
            'location' => 'Nakuru, Kenya',
            'status' => 'Available',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Minor front grill and light damage',
            'description' => 'Well maintained unit with clean interior and good suspension.',
            'image_url' => 'https://images.unsplash.com/photo-1549924231-f129b911e442?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1519641471654-76ce0107ad1b?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1563720223185-11003d516935?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1525609004556-c46c7d6cf023?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Subaru',
            'model' => 'Forester',
            'year' => 2016,
            'vin' => 'JF2SJADC9GH451102',
            'mileage' => 104900,
            'price' => 5200,
            'location' => 'Eldoret, Kenya',
            'status' => 'Available',
            'title_status' => 'Salvage',
            'damage_summary' => 'Left rear quarter panel dent',
            'description' => 'AWD drivetrain intact, engine starts and idles well.',
            'image_url' => 'https://images.unsplash.com/photo-1494976688153-ca5c6f5b6f5f?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1492144534655-ae79c964c9d7?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1511919884226-fd3cad34687c?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1549399542-7e3f8b79c341?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Mazda',
            'model' => 'Demio',
            'year' => 2015,
            'vin' => 'JM1DJ2A58F0123103',
            'mileage' => 112500,
            'price' => 3100,
            'location' => 'Thika, Kenya',
            'status' => 'Available',
            'title_status' => 'Salvage',
            'damage_summary' => 'Bonnet and bumper scratches',
            'description' => 'Fuel efficient hatchback with healthy gearbox.',
            'image_url' => 'https://images.unsplash.com/photo-1449965408869-eaa3f722e40d?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1502877338535-766e1452684a?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1494905998402-395d579af36f?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1493238792000-8113da705763?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Volkswagen',
            'model' => 'Golf',
            'year' => 2018,
            'vin' => 'WVWZZZAUZJW451104',
            'mileage' => 68800,
            'price' => 6400,
            'location' => 'Machakos, Kenya',
            'status' => 'Pending',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Rear hatch dent and taillight crack',
            'description' => 'Turbo engine responsive, AC and infotainment functional.',
            'image_url' => 'https://images.unsplash.com/photo-1533473359331-0135ef1b58bf?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1532581140115-3e355d1ed1de?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1489824904134-891ab64532f1?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1503376780353-7e6692767b70?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Mitsubishi',
            'model' => 'Outlander',
            'year' => 2021,
            'vin' => 'JA4AZ3A37MZ551105',
            'mileage' => 55200,
            'price' => 9800,
            'location' => 'Nyeri, Kenya',
            'status' => 'Available',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Front right fender repair',
            'description' => 'Hybrid system inspected, road-test complete.',
            'image_url' => 'https://images.unsplash.com/photo-1542282088-fe8426682b8f?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1517673132405-a56a62b18caf?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1525609004556-c46c7d6cf023?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1563720223185-11003d516935?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'BMW',
            'model' => '320i',
            'year' => 2014,
            'vin' => 'WBA3B1C54EF661106',
            'mileage' => 127400,
            'price' => 5900,
            'location' => 'Kisii, Kenya',
            'status' => 'Available',
            'title_status' => 'Salvage',
            'damage_summary' => 'Front suspension arm replacement needed',
            'description' => 'Engine smooth, minor dashboard trim wear.',
            'image_url' => 'https://images.unsplash.com/photo-1555215695-3004980ad54e?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1492144534655-ae79c964c9d7?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1511919884226-fd3cad34687c?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1532581140115-3e355d1ed1de?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Mercedes-Benz',
            'model' => 'C200',
            'year' => 2017,
            'vin' => 'WDDWF4KB8HR771107',
            'mileage' => 98100,
            'price' => 8900,
            'location' => 'Naivasha, Kenya',
            'status' => 'Pending',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Rear bumper repaint',
            'description' => 'Clean cabin and responsive transmission shifts.',
            'image_url' => 'https://images.unsplash.com/photo-1617469767053-d3b523a0b982?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1549399542-7e3f8b79c341?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1533473359331-0135ef1b58bf?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1494905998402-395d579af36f?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Suzuki',
            'model' => 'Swift',
            'year' => 2019,
            'vin' => 'JS2ZC31S0K6311108',
            'mileage' => 61500,
            'price' => 4700,
            'location' => 'Meru, Kenya',
            'status' => 'Available',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Door skin replacement on passenger side',
            'description' => 'Compact city car with economical fuel use.',
            'image_url' => 'https://images.unsplash.com/photo-1489824904134-891ab64532f1?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1503376780353-7e6692767b70?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1493238792000-8113da705763?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1494976388531-d1058494cdd8?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Isuzu',
            'model' => 'D-Max',
            'year' => 2022,
            'vin' => 'MPATFS87JNT881109',
            'mileage' => 43800,
            'price' => 11200,
            'location' => 'Kitale, Kenya',
            'status' => 'Available',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Tailgate and rear lamp assembly replaced',
            'description' => 'Strong chassis, suitable for heavy-duty tasks.',
            'image_url' => 'https://images.unsplash.com/photo-1605559424843-9e4c228bf1c2?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1519641471654-76ce0107ad1b?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1485291571150-772bcfc10da5?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1525609004556-c46c7d6cf023?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Toyota',
            'model' => 'RAV4',
            'year' => 2021,
            'vin' => 'JTMWFREV6MJ121110',
            'mileage' => 50900,
            'price' => 10800,
            'location' => 'Malindi, Kenya',
            'status' => 'Available',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Minor bonnet alignment correction',
            'description' => 'Popular SUV with excellent handling and comfort.',
            'image_url' => 'https://images.unsplash.com/photo-1519681393784-d120267933ba?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1542282088-fe8426682b8f?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1563720223185-11003d516935?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1492144534655-ae79c964c9d7?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Honda',
            'model' => 'Fit',
            'year' => 2016,
            'vin' => 'JHMGK5H76GX331111',
            'mileage' => 99500,
            'price' => 3600,
            'location' => 'Kakamega, Kenya',
            'status' => 'Pending',
            'title_status' => 'Salvage',
            'damage_summary' => 'Front bumper clips and fog light missing',
            'description' => 'Reliable compact with roomy interior layout.',
            'image_url' => 'https://images.unsplash.com/photo-1494905998402-395d579af36f?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1449965408869-eaa3f722e40d?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1502877338535-766e1452684a?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1493238792000-8113da705763?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
        [
            'make' => 'Nissan',
            'model' => 'Note',
            'year' => 2018,
            'vin' => 'SJNFAAZE0U6112112',
            'mileage' => 74100,
            'price' => 4200,
            'location' => 'Kericho, Kenya',
            'status' => 'Available',
            'title_status' => 'Rebuilt',
            'damage_summary' => 'Rear spoiler and hatch lock replaced',
            'description' => 'Smooth CVT response and efficient daily runner.',
            'image_url' => 'https://images.unsplash.com/photo-1503736334956-4c8f8e92946d?auto=format&fit=crop&w=1200&q=80',
            'images' => [
                'https://images.unsplash.com/photo-1532581140115-3e355d1ed1de?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1489824904134-891ab64532f1?auto=format&fit=crop&w=1200&q=80',
                'https://images.unsplash.com/photo-1549399542-7e3f8b79c341?auto=format&fit=crop&w=1200&q=80',
            ],
        ],
    ];
}

function populateSeedImageFilesForExistingCars(PDO $pdo): void
{
    $seedByVin = [];
    foreach (seedCars() as $seed) {
        $seedByVin[$seed['vin']] = $seed;
    }

    $cars = $pdo->query('SELECT id, vin, image_url FROM cars')->fetchAll();
    $updateCarStmt = $pdo->prepare('UPDATE cars SET image_url = :image_url WHERE id = :id');
    $countImagesStmt = $pdo->prepare('SELECT COUNT(*) FROM car_images WHERE car_id = :car_id');
    $insertImageStmt = $pdo->prepare('INSERT INTO car_images (car_id, url, sort_order) VALUES (:car_id, :url, :sort_order)');

    foreach ($cars as $car) {
        $vin = (string)$car['vin'];
        if (!isset($seedByVin[$vin])) {
            continue;
        }

        $carId = (int)$car['id'];
        $currentImageUrl = (string)$car['image_url'];

        if (!isLocalImagePath($currentImageUrl)) {
            $updateCarStmt->execute([
                ':id' => $carId,
                ':image_url' => storeSeedImageOrFallback($seedByVin[$vin]['image_url']),
            ]);
        }

        $countImagesStmt->execute([':car_id' => $carId]);
        $hasImages = ((int)$countImagesStmt->fetchColumn()) > 0;
        if ($hasImages) {
            continue;
        }

        foreach ($seedByVin[$vin]['images'] as $index => $url) {
            $insertImageStmt->execute([
                ':car_id' => $carId,
                ':url' => storeSeedImageOrFallback($url),
                ':sort_order' => $index,
            ]);
        }
    }
}

function storeSeedImageOrFallback(string $source): string
{
    $created = [];

    try {
        return materializeImageReference($source, $created);
    } catch (RuntimeException) {
        return ensureDefaultPlaceholderImage();
    }
}

function ensureDefaultPlaceholderImage(): string
{
    ensureImagesDirectory();

    $absolutePath = __DIR__ . '/' . DEFAULT_IMAGE_PATH;
    if (is_file($absolutePath)) {
        return DEFAULT_IMAGE_PATH;
    }

    $placeholder = base64_decode('/9j/4AAQSkZJRgABAQAAAQABAAD/2wCEAAkGBxAQEBUQEBAVFRAQEA8QDxAQEA8QFRAVFhUVFRUYHSggGBolGxUVITEhJSkrLi4uFx8zODMsNygtLisBCgoKDg0OFQ8QFSsdFR0rKysrKysrKysrKysrKysrKysrKysrKysrKysrKysrKysrKysrKysrKysrKysrK//AABEIAAEAAQMBIgACEQEDEQH/xAAXAAADAQAAAAAAAAAAAAAAAAAAAQID/8QAFBABAAAAAAAAAAAAAAAAAAAAAP/aAAwDAQACEAMQAAAB6gD/xAAZEAEAAgMAAAAAAAAAAAAAAAABABEhMUH/2gAIAQEAAT8Aq2qv/8QAFBEBAAAAAAAAAAAAAAAAAAAAAP/aAAgBAgEBPwD/xAAUEQEAAAAAAAAAAAAAAAAAAAAA/9oACAEDAQE/AT//2Q==', true);
    if ($placeholder === false || @file_put_contents($absolutePath, $placeholder) === false) {
        throw new RuntimeException('Failed to create placeholder image file.');
    }

    return DEFAULT_IMAGE_PATH;
}

function tableExists(PDO $pdo, string $table): bool
{
    $stmt = $pdo->prepare("SELECT name FROM sqlite_master WHERE type = 'table' AND name = :name LIMIT 1");
    $stmt->execute([':name' => $table]);

    return (bool)$stmt->fetchColumn();
}

function columnExists(PDO $pdo, string $table, string $column): bool
{
    $stmt = $pdo->query('PRAGMA table_info(' . $table . ')');
    $columns = $stmt->fetchAll();

    foreach ($columns as $info) {
        if (($info['name'] ?? null) === $column) {
            return true;
        }
    }

    return false;
}
