<?php

declare(strict_types=1);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/helpers.php';

setupCors();
$pdo = db();
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($id <= 0) {
    jsonResponse(['message' => 'A valid car id is required'], 422);
}

if ($method === 'GET') {
    $car = fetchCar($pdo, $id);

    if (!$car) {
        jsonResponse(['message' => 'Car not found'], 404);
    }

    jsonResponse(['car' => $car]);
}

if ($method === 'PUT' || $method === 'PATCH') {
    requireAdmin($pdo);

    $existingCar = fetchCar($pdo, $id);
    if (!$existingCar) {
        jsonResponse(['message' => 'Car not found'], 404);
    }

    $request = parseCarWriteRequest($method);
    $body = $request['body'];
    $files = $request['files'];

    $coverUpload = getPreferredCoverUpload($files);
    $galleryUploads = getGalleryUploads($files);

    $payload = validateUpdateCarPayload($body, $coverUpload !== null, $galleryUploads !== []);

    $updatedCarData = [
        'make' => array_key_exists('make', $payload) ? $payload['make'] : $existingCar['make'],
        'model' => array_key_exists('model', $payload) ? $payload['model'] : $existingCar['model'],
        'year' => array_key_exists('year', $payload) ? $payload['year'] : (int)$existingCar['year'],
        'vin' => array_key_exists('vin', $payload) ? $payload['vin'] : $existingCar['vin'],
        'mileage' => array_key_exists('mileage', $payload) ? $payload['mileage'] : (int)$existingCar['mileage'],
        'price' => array_key_exists('price', $payload) ? $payload['price'] : (float)$existingCar['price'],
        'location' => array_key_exists('location', $payload) ? $payload['location'] : $existingCar['location'],
        'status' => array_key_exists('status', $payload) ? $payload['status'] : $existingCar['status'],
        'title_status' => array_key_exists('title_status', $payload) ? $payload['title_status'] : $existingCar['title_status'],
        'damage_summary' => array_key_exists('damage_summary', $payload) ? $payload['damage_summary'] : $existingCar['damage_summary'],
        'description' => array_key_exists('description', $payload) ? $payload['description'] : $existingCar['description'],
        'image_url' => $existingCar['image_url'],
    ];

    $replaceGallery = false;
    $newGalleryImages = [];
    $createdFiles = [];

    try {
        if ($coverUpload !== null) {
            $updatedCarData['image_url'] = materializeUploadedFile($coverUpload, $createdFiles);
        } elseif (array_key_exists('image_url', $payload)) {
            $updatedCarData['image_url'] = materializeImageReference((string)$payload['image_url'], $createdFiles);
        }

        if ($galleryUploads !== []) {
            $replaceGallery = true;
            $newGalleryImages = materializeUploadedImages($galleryUploads, $createdFiles);
        } elseif ($payload['_images_provided']) {
            $replaceGallery = true;
            $newGalleryImages = materializeImageReferenceList($payload['images'], $createdFiles);
        }

        $oldImagePaths = collectCarImagePaths($existingCar);

        $stmt = $pdo->prepare('UPDATE cars
          SET make = :make,
              model = :model,
              year = :year,
              vin = :vin,
              mileage = :mileage,
              price = :price,
              location = :location,
              status = :status,
              title_status = :title_status,
              damage_summary = :damage_summary,
              description = :description,
              image_url = :image_url,
              updated_at = :updated_at
          WHERE id = :id');

        $pdo->beginTransaction();

        $stmt->execute([
            ':make' => $updatedCarData['make'],
            ':model' => $updatedCarData['model'],
            ':year' => $updatedCarData['year'],
            ':vin' => $updatedCarData['vin'],
            ':mileage' => $updatedCarData['mileage'],
            ':price' => $updatedCarData['price'],
            ':location' => $updatedCarData['location'],
            ':status' => $updatedCarData['status'],
            ':title_status' => $updatedCarData['title_status'],
            ':damage_summary' => $updatedCarData['damage_summary'],
            ':description' => $updatedCarData['description'],
            ':image_url' => $updatedCarData['image_url'],
            ':updated_at' => gmdate('Y-m-d H:i:s'),
            ':id' => $id,
        ]);

        if ($replaceGallery) {
            replaceCarImages($pdo, $id, $newGalleryImages);
        }

        $pdo->commit();

        $car = fetchCar($pdo, $id);
        $newImagePaths = collectCarImagePaths($car);
        $pathsToDelete = array_values(array_diff($oldImagePaths, $newImagePaths));
        deleteUnreferencedLocalImageFiles($pdo, $pathsToDelete);
    } catch (RuntimeException $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }

        deleteLocalImageFiles($createdFiles);
        jsonResponse([
            'message' => 'Validation failed',
            'errors' => ['images' => [$e->getMessage()]],
        ], 422);
    } catch (PDOException $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }

        deleteLocalImageFiles($createdFiles);
        jsonResponse(['message' => 'Failed to update car: ' . $e->getMessage()], 400);
    }

    jsonResponse(['car' => $car]);
}

if ($method === 'DELETE') {
    requireAdmin($pdo);

    $car = fetchCar($pdo, $id);
    if (!$car) {
        jsonResponse(['message' => 'Car not found'], 404);
    }

    $imagePaths = collectCarImagePaths($car);

    $stmt = $pdo->prepare('DELETE FROM cars WHERE id = :id');
    $stmt->execute([':id' => $id]);

    deleteUnreferencedLocalImageFiles($pdo, $imagePaths);

    jsonResponse(['message' => 'Car deleted']);
}

jsonResponse(['message' => 'Method not allowed'], 405);
